<?php

class Transaction
{
    private PDO $pdo;

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /* =========================
       CORE SUMMARIES
    ========================= */

    public function summary(string $start = null, string $end = null): array
    {
        $where = '';
        $params = [];

        if ($start && $end) {
            $where = "WHERE transaction_date BETWEEN ? AND ?";
            $params = [$start, $end];
        }

        $stmt = $this->pdo->prepare("
            SELECT
                COALESCE(SUM(CASE WHEN type='income' THEN amount END),0) AS income,
                COALESCE(SUM(CASE WHEN type='expense' THEN amount END),0) AS expense
            FROM transactions
            $where
        ");
        $stmt->execute($params);

        return $stmt->fetch();
    }

    public function today(): array
    {
        $today = date('Y-m-d');
        return $this->summary($today, $today);
    }

    public function month(): array
    {
        return $this->summary(date('Y-m-01'), date('Y-m-t'));
    }

    public function year(): array
    {
        return $this->summary(date('Y-01-01'), date('Y-12-31'));
    }

    public function range(string $start, string $end): array
    {
        return $this->summary($start, $end);
    }

    /* =========================
       TRANSACTION HISTORY
    ========================= */

    public function allWithBalance(): array
    {
        // Oldest → newest for correct balance
        $rows = $this->pdo->query("
            SELECT *
            FROM transactions
            ORDER BY transaction_date ASC, id ASC
        ")->fetchAll();

        $balance = 0;

        foreach ($rows as &$r) {
            if ($r['type'] === 'income') {
                $balance += (float) $r['amount'];
            } else {
                $balance -= (float) $r['amount'];
            }

            // Always attach running balance
            $r['running_balance'] = $balance;
        }

        // Newest first for UI
        return array_reverse($rows);
    }

    /* =========================
       ADD TRANSACTION
    ========================= */

    public function add(array $data): void
    {
        $stmt = $this->pdo->prepare("
            INSERT INTO transactions
            (type, amount, category, description, transaction_date)
            VALUES (?, ?, ?, ?, ?)
        ");

        $stmt->execute([
            $data['type'],
            (float) $data['amount'],
            $data['category'] ?? null,
            $data['description'] ?? null,
            $data['date']
        ]);
    }

    public function last7Days(): array
    {
        $stmt = $this->pdo->query("
        SELECT transaction_date,
        SUM(CASE WHEN type='income' THEN amount ELSE 0 END) income,
        SUM(CASE WHEN type='expense' THEN amount ELSE 0 END) expense
        FROM transactions
        WHERE transaction_date >= CURDATE() - INTERVAL 6 DAY
        GROUP BY transaction_date
        ORDER BY transaction_date
    ");

        return $stmt->fetchAll();
    }

    public function reportByDate($date)
    {
        $stmt = $this->pdo->prepare("
        SELECT * FROM transactions
        WHERE transaction_date = ?
        ORDER BY transaction_date DESC, id DESC
    ");
        $stmt->execute([$date]);
        return $stmt->fetchAll();
    }

    public function reportByMonth($month)
    {
        $stmt = $this->pdo->prepare("
        SELECT * FROM transactions
        WHERE DATE_FORMAT(transaction_date, '%Y-%m') = ?
        ORDER BY transaction_date DESC, id DESC
    ");
        $stmt->execute([$month]);
        return $stmt->fetchAll();
    }

    public function reportByYear($year)
    {
        $stmt = $this->pdo->prepare("
        SELECT * FROM transactions
        WHERE YEAR(transaction_date) = ?
        ORDER BY transaction_date DESC, id DESC
    ");
        $stmt->execute([$year]);
        return $stmt->fetchAll();
    }

    public function reportByRange($start, $end)
    {
        $stmt = $this->pdo->prepare("
        SELECT * FROM transactions
        WHERE transaction_date BETWEEN ? AND ?
        ORDER BY transaction_date DESC, id DESC
    ");
        $stmt->execute([$start, $end]);
        return $stmt->fetchAll();
    }
}